/// <summary>
/// FancyBarTimer by tobbe. Please use the NinjaTrader support forum and private messaging
/// if you find a bug or have any questions regarding this indicator. The logic
/// used to decide session time, market connections etc is from the stock NT bartimer.
/// 
/// 2008-04-28	Release 1.0.0.
/// 
/// 2008-05-02	Release 1.1.0.
/// 	- Display bartimer even if there is no market feed.
/// 	- Added support for volume bars.
/// 
/// 2009-02-08	Release 1.1.1.
/// 	- Changed names of properties.
/// 
/// 2009-08-17	Release 1.1.2.
/// 	- Fixed bug reported by BlowFish. The bar timer will now only draw minute
///       markers if there is space for them.
/// 
/// 2009-08-20	Release 1.1.3.
/// 	- Fixed bug reported by dwalls. Bars.PercentComplete does not seem to work
///       for volume bars.
/// </summary>

#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

#region FancyBarTimerPosition
/// <summary>
/// Custom enum for the placement of the bar timer. Custom enums must be
/// placed in the global namespace to compile. It would have been nicer
/// to have it nested in the FancyBarTimer class.
/// </summary>
public enum FancyBarTimerPosition
{
	TopLeft,
	TopCenter,
	TopRight,
	BottomLeft,
	BottomCenter,
	BottomRight
}
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// A fancy bar timer
    /// </summary>
    [Description("A fancy bar timer")]
    public class FancyBarTimer : Indicator
    {
        #region Variables
        // Wizard generated variables
        // User defined variables (add any user defined variables below)
				
		// Default position of the fancy bar timer on the chart.
		private FancyBarTimerPosition position = FancyBarTimerPosition.BottomRight;
		
		// Allow the user to move the bartimer from the sides of the chart.
		// The meaning of these variables depends on the chosen position.
		// For example, if the position is TopLeft, the bar will be moved
		// out to the right and down. If the position is BottomRight, the
		// bar will be moved to the left and up.
		private int horizontalOffset = 0;
		private int verticalOffset = 0;
		
		// The opacity to use.
		private int alphaPercent = 50;
		
		// Bar timer dimensions.
		private int barWidth = 150;
		private int barHeight = 10;

		// Keep a local copy of brushes and pens for performance.
		private SolidBrush barBrush = new SolidBrush(Color.FromArgb(127, Color.Violet)); 
		private Pen framePen = new Pen(Color.Gray);
		
		// The timer used to trigger redraws.
		private System.Windows.Forms.Timer timer;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            CalculateOnBarClose	= true;
			ChartOnly			= true;
            Overlay				= true;
            PriceTypeSupported	= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			// Create and start timer to run once every second. The timer
			// is not needed if the chart is set to display volume bars.
			if (timer == null && DisplayTime() && !DelayedData() && MinuteBars())
			{
				timer = new System.Windows.Forms.Timer();
				timer.Interval = 1000;
				timer.Tick += OnFancyBarTimerTick;
				timer.Enabled = true;
			}
        }
		
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max) {
			bool displayTime = DisplayTime();
			
			if ((MinuteBars() || VolumeBars()))
			{
				// First calculate all bounds needed to draw the bar.
				int x = 0;
				int y = OnTop() ? bounds.Y + verticalOffset : bounds.Height - barHeight - verticalOffset;
				
				if (ToTheLeft())
				{
					x = bounds.X + horizontalOffset;					
				}
				else if (Centered())
				{
					x = (bounds.Width - barWidth)/2;					
				}
				else if (ToTheRight())
				{
					x = bounds.Width - barWidth - horizontalOffset;
				}

				if (displayTime)
				{			
					// Display the bartime if we're connected to a feed (market or playback).
					// Bars.PercentComplete does not work with volume bars in the current version of NT.
					double percentComplete = VolumeBars() ? (double)Bars.Get(Bars.Count - 1).Volume / (double)Bars.Period.Value : Bars.PercentComplete;
					int w = (int)((1 - percentComplete)*(double)barWidth);
					graphics.FillRectangle(barBrush, x+barWidth-w, y, w, barHeight);
				}
				graphics.DrawRectangle(framePen, x, y, barWidth, barHeight);
				
				if (MinuteBars())
				{
					// Draw vertical minute lines if there is enough space for them.
					int minuteWidth = barWidth / Bars.Period.Value;

					if (minuteWidth > 1)
					{
						int minuteX = x + minuteWidth;
						framePen.DashStyle = DashStyle.Dash;
						
						if (minuteWidth > 0)
						while (minuteX < x + barWidth)
						{
							graphics.DrawLine(framePen, minuteX, y, minuteX, y+barHeight);
							minuteX += minuteWidth;
						}
						framePen.DashStyle = DashStyle.Solid;
					}
				}
			}
			
			if (!displayTime && timer != null)
			{
				// Not displaying anymore, stop the timer and make sure it gets disposed.
				timer.Enabled = false;
				timer = null;
			}
		}		

		#region Properties
		[Browsable(true)]
		[Description("The position of the fancy bar timer on the chart")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Position")]
		public FancyBarTimerPosition Position
		{
			get { return position; }
			set { position = value; }
		}

		[Browsable(true)]
		[Description("The width of the fancy bar timer in pixels")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Width")]
		public int BarWidth
		{
			get { return barWidth; }
			// Min width is 50 pixels, max width is 1000 pixels.
			set	{ barWidth = Math.Max(50, Math.Min(1000, value)); }
		}

		[Browsable(true)]
		[Description("The height of the fancy bar timer in pixels")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Height")]
		public int BarHeight
		{
			get { return barHeight; }
			// Min height is 5 pixels, max height is 500 pixels.
			set	{ barHeight = Math.Max(5, Math.Min(500, value)); }
		}

		[Browsable(true)]
		[Description("An offset from the left or right side in pixels")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("HorizontalOffset")]
		public int LeftRightOffset
		{
			get { return horizontalOffset; }
			set	{ horizontalOffset = Math.Max(0, Math.Min(250, value)); }
		}

		[Browsable(true)]
		[Description("An offset from the top or bottom in pixels")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("VerticalOffset")]
		public int VerticalOffset
		{
			get { return verticalOffset; }
			set	{ verticalOffset = Math.Max(0, Math.Min(250, value)); }
		}
		
		[Browsable(true)]
		[Description("The alpha (opacity) in percent for the fancy bar timer color")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("AlphaPercent")]
		public int AlphaPercent
		{
			get { return alphaPercent; }
			set
			{
				alphaPercent = Math.Max(0, Math.Min(100, value));
				Color barAlphaColor = Color.FromArgb((int)((double)alphaPercent/100*(double)255),barBrush.Color);
				barBrush = new SolidBrush(barAlphaColor);
			}
		}

		[Browsable(true)]
		[XmlIgnore()]
		[Description("The color of the fancy bar timer frame")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("FrameColor")]
		public Color FrameColor
		{
			get { return framePen.Color; }
			set { framePen = new Pen(value); }
		}

		[Browsable(false)]
		public string FrameColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(framePen.Color); }
			set { framePen = new Pen(NinjaTrader.Gui.Design.SerializableColor.FromString(value)); }
		}

		[Browsable(true)]
		[XmlIgnore()]
		[Description("The color of the fancy bar timer")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("BarColor")]
		public Color BarColor
		{
			get { return barBrush.Color; }
			set
			{
				Color barAlphaColor = Color.FromArgb((int)((double)alphaPercent/100*(double)255),value);
				barBrush = new SolidBrush(barAlphaColor);
			}
		}

		[Browsable(false)]
		public string BarColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barBrush.Color); }
			set
			{ 
				Color barColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value);
				Color barAlphaColor = Color.FromArgb((int)((double)alphaPercent/100*(double)255),barColor);
				barBrush = new SolidBrush(barAlphaColor);
			}
		}
        #endregion
		
		#region FancyMethods
		private void OnFancyBarTimerTick(object sender, EventArgs e)
        {
			if (DisplayTime())
			{
				ChartControl.ChartPanel.Invalidate();
			}
        }
		
		private bool MinuteBars() 
		{
			return Bars.Period.Id == PeriodType.Minute;
		}
		
		private bool VolumeBars() 
		{
			return Bars.Period.Id == PeriodType.Volume;
		}

		private bool OnTop()
		{
			return position == FancyBarTimerPosition.TopLeft ||
				   position == FancyBarTimerPosition.TopCenter ||
				   position == FancyBarTimerPosition.TopRight;
		}
		
		private bool ToTheLeft()
		{
			return position == FancyBarTimerPosition.BottomLeft || position == FancyBarTimerPosition.TopLeft;
		}
		
		
		private bool Centered()
		{
			return position == FancyBarTimerPosition.BottomCenter || position == FancyBarTimerPosition.TopCenter;
		}
		
		private bool ToTheRight()
		{
			return position == FancyBarTimerPosition.BottomRight || position == FancyBarTimerPosition.TopRight;
		}

		private bool DelayedData()
		{
			return Bars.MarketData != null &&
				   Bars.MarketData.Connection.Options.Provider == Cbi.Provider.OpenTick &&
				  (Bars.MarketData.Connection.Options as Cbi.OpenTickOptions).UseDelayedData;
		}
		#endregion

		#region FromOriginalNinjaTraderBarTimer
		/// <summary>
		/// These methods are from the BarTimer that is supplied with NT.
		/// </summary>
		/// <returns>true if the bar timer should be displayed (connected, in session etc)</returns>
        private bool DisplayTime()
        {
			if (ChartControl != null
					&& Bars != null
					&& Bars.Count > 0
					&& Bars.MarketData != null
					&& Bars.MarketData.Connection.PriceStatus == Cbi.ConnectionStatus.Connected
					&& (Bars.MarketData.Connection.Options.Provider != Cbi.Provider.OpenTick || !(Bars.MarketData.Connection.Options as Cbi.OpenTickOptions).UseDelayedData)
					&& Bars.Session.InSession(Now, Bars.Period.Id, true))
				return true;

            return false;
        }

		private DateTime Now
		{
			get { return (Bars.MarketData.Connection.Options.Provider == Cbi.Provider.Replay ? Bars.MarketData.Connection.Now : DateTime.Now); }
		}

        public override void Dispose()
        {
            if (timer != null)
            {
                timer.Enabled = false;
                timer = null;
            }
            base.Dispose();
        }
		#endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private FancyBarTimer[] cacheFancyBarTimer = null;

        private static FancyBarTimer checkFancyBarTimer = new FancyBarTimer();

        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        public FancyBarTimer FancyBarTimer(int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            return FancyBarTimer(Input, barHeight, barWidth, leftRightOffset, position, verticalOffset);
        }

        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        public FancyBarTimer FancyBarTimer(Data.IDataSeries input, int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            checkFancyBarTimer.BarHeight = barHeight;
            barHeight = checkFancyBarTimer.BarHeight;
            checkFancyBarTimer.BarWidth = barWidth;
            barWidth = checkFancyBarTimer.BarWidth;
            checkFancyBarTimer.LeftRightOffset = leftRightOffset;
            leftRightOffset = checkFancyBarTimer.LeftRightOffset;
            checkFancyBarTimer.Position = position;
            position = checkFancyBarTimer.Position;
            checkFancyBarTimer.VerticalOffset = verticalOffset;
            verticalOffset = checkFancyBarTimer.VerticalOffset;

            if (cacheFancyBarTimer != null)
                for (int idx = 0; idx < cacheFancyBarTimer.Length; idx++)
                    if (cacheFancyBarTimer[idx].BarHeight == barHeight && cacheFancyBarTimer[idx].BarWidth == barWidth && cacheFancyBarTimer[idx].LeftRightOffset == leftRightOffset && cacheFancyBarTimer[idx].Position == position && cacheFancyBarTimer[idx].VerticalOffset == verticalOffset && cacheFancyBarTimer[idx].EqualsInput(input))
                        return cacheFancyBarTimer[idx];

            FancyBarTimer indicator = new FancyBarTimer();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.BarHeight = barHeight;
            indicator.BarWidth = barWidth;
            indicator.LeftRightOffset = leftRightOffset;
            indicator.Position = position;
            indicator.VerticalOffset = verticalOffset;
            indicator.SetUp();

            FancyBarTimer[] tmp = new FancyBarTimer[cacheFancyBarTimer == null ? 1 : cacheFancyBarTimer.Length + 1];
            if (cacheFancyBarTimer != null)
                cacheFancyBarTimer.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheFancyBarTimer = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.FancyBarTimer FancyBarTimer(int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            return _indicator.FancyBarTimer(Input, barHeight, barWidth, leftRightOffset, position, verticalOffset);
        }

        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        public Indicator.FancyBarTimer FancyBarTimer(Data.IDataSeries input, int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            return _indicator.FancyBarTimer(input, barHeight, barWidth, leftRightOffset, position, verticalOffset);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.FancyBarTimer FancyBarTimer(int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            return _indicator.FancyBarTimer(Input, barHeight, barWidth, leftRightOffset, position, verticalOffset);
        }

        /// <summary>
        /// A fancy bar timer
        /// </summary>
        /// <returns></returns>
        public Indicator.FancyBarTimer FancyBarTimer(Data.IDataSeries input, int barHeight, int barWidth, int leftRightOffset, FancyBarTimerPosition position, int verticalOffset)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.FancyBarTimer(input, barHeight, barWidth, leftRightOffset, position, verticalOffset);
        }

    }
}
#endregion
